// Copyright 2015 The Bazel Authors. All rights reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package com.google.devtools.build.buildjar.genclass;

import static com.google.common.collect.ImmutableSet.toImmutableSet;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableSet;
import com.google.devtools.build.buildjar.jarhelper.JarCreator;
import com.google.devtools.build.buildjar.proto.JavaCompilation.CompilationUnit;
import com.google.devtools.build.buildjar.proto.JavaCompilation.Manifest;
import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.function.Predicate;
import java.util.jar.JarFile;
import java.util.stream.Stream;

/**
 * GenClass post-processes the output of a Java compilation, and produces a jar containing only the
 * class files for sources that were generated by annotation processors.
 */
public class GenClass {

  public static void main(String[] args) throws IOException {
    GenClassOptions options = GenClassOptionsParser.parse(Arrays.asList(args));
    Manifest manifest = readManifest(options.manifest());
    Map<String, byte[]> classes = extractGeneratedClasses(options.classJar(), manifest);
    writeOutputJar(classes, options);
  }

  /** Reads the compilation manifest. */
  private static Manifest readManifest(Path path) throws IOException {
    Manifest manifest;
    try (InputStream inputStream = Files.newInputStream(path)) {
      manifest = Manifest.parseFrom(inputStream);
    }
    return manifest;
  }

  /**
   * For each top-level class in the compilation matching the given predicate, determine the path
   * prefix of classes corresponding to that compilation unit.
   */
  @VisibleForTesting
  static ImmutableSet<String> getPrefixes(Manifest manifest, Predicate<CompilationUnit> p) {
    return manifest.getCompilationUnitList().stream()
        .filter(p)
        .flatMap(unit -> getUnitPrefixes(unit))
        .collect(toImmutableSet());
  }

  /**
   * Prefixes are used to correctly handle inner classes, e.g. the top-level class "c.g.Foo" may
   * correspond to "c/g/Foo.class" and also "c/g/Foo$Inner.class" or "c/g/Foo$0.class".
   */
  private static Stream<String> getUnitPrefixes(CompilationUnit unit) {
    String pkg;
    if (unit.hasPkg()) {
      pkg = unit.getPkg().replace('.', '/') + "/";
    } else {
      pkg = "";
    }
    return unit.getTopLevelList().stream().map(toplevel -> pkg + toplevel);
  }

  /**
   * Unzip all the class files that correspond to annotation processor- generated sources into the
   * temporary directory.
   */
  private static Map<String, byte[]> extractGeneratedClasses(Path classJar, Manifest manifest)
      throws IOException {
    Map<String, byte[]> classes = new HashMap<>();
    ImmutableSet<String> generatedFilePrefixes =
        getPrefixes(manifest, unit -> unit.getGeneratedByAnnotationProcessor());
    ImmutableSet<String> userWrittenFilePrefixes =
        getPrefixes(manifest, unit -> !unit.getGeneratedByAnnotationProcessor());
    try (JarFile jar = new JarFile(classJar.toFile())) {
      jar.stream()
          .filter(entry -> entry.getName().endsWith(".class"))
          .forEach(
              entry -> {
                String name = entry.getName();
                String className = name.substring(0, name.length() - ".class".length());
                if (prefixesContains(generatedFilePrefixes, className)
                    // Assume that prefixes that don't correspond to a known hand-written source are
                    // generated.
                    || !prefixesContains(userWrittenFilePrefixes, className)) {
                  try (InputStream is = jar.getInputStream(entry)) {
                    classes.put(name, is.readAllBytes());
                  } catch (IOException e) {
                    throw new UncheckedIOException(e);
                  }
                }
              });
    }
    return classes;
  }

  /**
   * We want to include inner classes for generated source files, but a class whose name contains
   * '$' isn't necessarily an inner class. Check whether any prefix of the class name that ends with
   * '$' matches one of the top-level class names.
   */
  private static boolean prefixesContains(ImmutableSet<String> prefixes, String className) {
    if (prefixes.contains(className)) {
      return true;
    }
    for (int i = className.indexOf('$'); i != -1; i = className.indexOf('$', i + 1)) {
      if (prefixes.contains(className.substring(0, i))) {
        return true;
      }
    }
    return false;
  }

  /** Writes the generated class files to the output jar. */
  private static void writeOutputJar(Map<String, byte[]> classes, GenClassOptions options)
      throws IOException {
    JarCreator output = new JarCreator(options.outputJar());
    output.setCompression(true);
    classes.forEach((name, content) -> output.addEntry(name, content));
    output.execute();
  }
}
