// Copyright 2017 The Bazel Authors. All rights reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//    http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
package com.google.devtools.build.lib.skyframe;

import com.google.common.collect.ImmutableList;
import com.google.devtools.build.lib.events.EventHandler;
import com.google.devtools.build.lib.packages.BuildFileName;
import com.google.devtools.build.lib.server.FailureDetails;
import com.google.devtools.build.lib.skyframe.DiffAwarenessManager.EvaluatingVersionDiff;
import com.google.devtools.build.lib.skyframe.PackageFunction.ActionOnFilesystemErrorCodeLoadingBzlFile;
import com.google.devtools.build.lib.skyframe.PackageFunction.ActionOnIOExceptionReadingBuildFile;
import com.google.devtools.build.lib.skyframe.PackageLookupFunction.CrossRepositoryLabelViolationStrategy;
import com.google.devtools.build.lib.skyframe.SkyframeExecutor.DiffCheckNotificationOptions;
import com.google.devtools.common.options.OptionsProvider;
import java.time.Duration;

/** Hardcoded constants describing bazel-on-skyframe behavior. */
public class BazelSkyframeExecutorConstants {
  private BazelSkyframeExecutorConstants() {}

  public static final CrossRepositoryLabelViolationStrategy
      CROSS_REPOSITORY_LABEL_VIOLATION_STRATEGY = CrossRepositoryLabelViolationStrategy.ERROR;

  public static final ImmutableList<BuildFileName> BUILD_FILES_BY_PRIORITY =
      ImmutableList.of(BuildFileName.BUILD_DOT_BAZEL, BuildFileName.BUILD);

  public static final ActionOnIOExceptionReadingBuildFile
      ACTION_ON_IO_EXCEPTION_READING_BUILD_FILE =
          ActionOnIOExceptionReadingBuildFile.UseOriginalIOException.INSTANCE;

  public static final ActionOnFilesystemErrorCodeLoadingBzlFile
      ACTION_ON_FILESYSTEM_ERROR_CODE_LOADING_BZL_FILE =
          filesystemCode -> filesystemCode == FailureDetails.Filesystem.Code.REMOTE_FILE_EVICTED;

  public static final boolean USE_REPO_DOT_BAZEL = true;

  public static final DiffCheckNotificationOptions DIFF_CHECK_NOTIFICATION_OPTIONS =
      new DiffCheckNotificationOptions() {
        @Override
        public boolean allowDiffCheck(
            EvaluatingVersionDiff versionDiff, EventHandler eventHandler, OptionsProvider options) {
          return true;
        }

        @Override
        public String getStatusMessage() {
          return "Checking for file changes...";
        }

        @Override
        public Duration getStatusUpdateDelay() {
          return Duration.ofSeconds(1);
        }
      };

  public static SequencedSkyframeExecutor.Builder newBazelSkyframeExecutorBuilder() {
    return SequencedSkyframeExecutor.builder()
        .setIgnoredSubdirectories(IgnoredSubdirectoriesFunction.INSTANCE)
        .setActionOnIOExceptionReadingBuildFile(ACTION_ON_IO_EXCEPTION_READING_BUILD_FILE)
        .setActionOnFilesystemErrorCodeLoadingBzlFile(
            ACTION_ON_FILESYSTEM_ERROR_CODE_LOADING_BZL_FILE)
        .setShouldUseRepoDotBazel(USE_REPO_DOT_BAZEL)
        .setCrossRepositoryLabelViolationStrategy(CROSS_REPOSITORY_LABEL_VIOLATION_STRATEGY)
        .setBuildFilesByPriority(BUILD_FILES_BY_PRIORITY)
        .setDiffCheckNotificationOptions(DIFF_CHECK_NOTIFICATION_OPTIONS);
  }
}
