// Copyright 2014 The Bazel Authors. All rights reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//    http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
package com.google.devtools.build.lib.skyframe;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import com.google.devtools.build.lib.actions.ActionAnalysisMetadata;
import com.google.devtools.build.lib.actions.Actions.GeneratingActions;
import com.google.devtools.build.lib.actions.Artifact;
import com.google.devtools.build.lib.actions.BasicActionLookupValue;
import com.google.devtools.build.lib.analysis.ConfiguredTarget;
import com.google.devtools.build.lib.cmdline.Label;
import com.google.devtools.build.lib.collect.nestedset.NestedSet;
import com.google.devtools.build.lib.concurrent.ThreadSafety.Immutable;
import com.google.devtools.build.lib.concurrent.ThreadSafety.ThreadSafe;
import com.google.devtools.build.lib.packages.Package;
import com.google.devtools.build.lib.skyframe.serialization.autocodec.AutoCodec;
import com.google.devtools.build.lib.skyframe.serialization.autocodec.AutoCodec.VisibleForSerialization;
import com.google.devtools.build.skyframe.SkyKey;
import java.util.ArrayList;
import javax.annotation.Nullable;

/** A non-rule configured target in the context of a Skyframe graph. */
@Immutable
@ThreadSafe
@AutoCodec
@VisibleForTesting
public final class NonRuleConfiguredTargetValue extends BasicActionLookupValue
    implements ConfiguredTargetValue {

  // These variables are only non-final because they may be clear()ed to save memory.
  // configuredTarget is null only after it is cleared.
  @Nullable private ConfiguredTarget configuredTarget;

  // May be null either after clearing or because transitive packages are not tracked.
  @Nullable private NestedSet<Package> transitivePackagesForPackageRootResolution;

  @AutoCodec.Instantiator
  @VisibleForSerialization
  NonRuleConfiguredTargetValue(
      ArrayList<ActionAnalysisMetadata> actions,
      ImmutableMap<Artifact, Integer> generatingActionIndex,
      ConfiguredTarget configuredTarget) {
    super(actions, generatingActionIndex, /*removeActionsAfterEvaluation=*/ false);
    this.configuredTarget = configuredTarget;
    // Transitive packages are not serialized.
    this.transitivePackagesForPackageRootResolution = null;
  }

  NonRuleConfiguredTargetValue(
      ConfiguredTarget configuredTarget,
      GeneratingActions generatingActions,
      @Nullable NestedSet<Package> transitivePackagesForPackageRootResolution,
      boolean removeActionsAfterEvaluation) {
    super(generatingActions, removeActionsAfterEvaluation);
    this.configuredTarget = Preconditions.checkNotNull(configuredTarget, generatingActions);
    this.transitivePackagesForPackageRootResolution = transitivePackagesForPackageRootResolution;
  }

  @VisibleForTesting
  @Override
  public ConfiguredTarget getConfiguredTarget() {
    Preconditions.checkNotNull(configuredTarget);
    return configuredTarget;
  }

  @VisibleForTesting
  @Override
  public ArrayList<ActionAnalysisMetadata> getActions() {
    Preconditions.checkNotNull(configuredTarget, this);
    return actions;
  }

  @Override
  public NestedSet<Package> getTransitivePackagesForPackageRootResolution() {
    return Preconditions.checkNotNull(transitivePackagesForPackageRootResolution);
  }

  @Override
  public void clear(boolean clearEverything) {
    Preconditions.checkNotNull(configuredTarget);
    Preconditions.checkNotNull(transitivePackagesForPackageRootResolution);
    if (clearEverything) {
      configuredTarget = null;
    }
    transitivePackagesForPackageRootResolution = null;
  }

  /**
   * Returns a label of NonRuleConfiguredTargetValue.
   */
  @ThreadSafe
  static Label extractLabel(SkyKey value) {
    Object valueName = value.argument();
    Preconditions.checkState(valueName instanceof ConfiguredTargetKey, valueName);
    return ((ConfiguredTargetKey) valueName).getLabel();
  }

  @Override
  public String toString() {
    return getStringHelper().add("configuredTarget", configuredTarget).toString();
  }
}
