// Copyright 2014 The Bazel Authors. All rights reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//    http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package com.google.devtools.build.lib.analysis.configuredtargets;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.devtools.build.lib.actions.Artifact;
import com.google.devtools.build.lib.actions.FilesetTraversalParams;
import com.google.devtools.build.lib.analysis.TargetContext;
import com.google.devtools.build.lib.analysis.TransitiveInfoCollection;
import com.google.devtools.build.lib.analysis.fileset.FilesetProvider;
import com.google.devtools.build.lib.cmdline.Label;
import com.google.devtools.build.lib.collect.nestedset.NestedSet;
import com.google.devtools.build.lib.packages.OutputFile;
import com.google.devtools.build.lib.packages.PackageSpecification.PackageGroupContents;
import com.google.devtools.build.lib.skyframe.BuildConfigurationValue;
import com.google.devtools.build.lib.skyframe.serialization.autocodec.AutoCodec;

/**
 * A configured target for output files generated by {@code Fileset} rules. They are almost the same
 * thing as output files except that they implement {@link FilesetProvider} so that {@code Fileset}
 * can figure out the link tree behind them.
 *
 * <p>In an ideal world, this would not be needed: Filesets would depend on other Filesets and not
 * their output directories. However, sometimes a Fileset depends on the output directory of another
 * Fileset. Thus, we need this hack.
 */
@AutoCodec
public final class FilesetOutputConfiguredTarget extends OutputFileConfiguredTarget
    implements FilesetProvider {
  private final ImmutableList<FilesetTraversalParams> filesetTraversals;

  public FilesetOutputConfiguredTarget(
      TargetContext targetContext,
      OutputFile outputFile,
      TransitiveInfoCollection generatingRule,
      Artifact outputArtifact,
      ImmutableList<FilesetTraversalParams> traversals) {
    this(
        targetContext.getLabel(),
        targetContext.getConfigurationKey(),
        targetContext.getVisibility(),
        generatingRule,
        outputArtifact,
        traversals);
    Preconditions.checkState(
        outputFile.getLabel().equals(targetContext.getLabel()),
        "mismatch: %s %s",
        outputFile,
        targetContext);
  }

  @AutoCodec.VisibleForSerialization
  @AutoCodec.Instantiator
  FilesetOutputConfiguredTarget(
      Label label,
      BuildConfigurationValue.Key configurationKey,
      NestedSet<PackageGroupContents> visibility,
      TransitiveInfoCollection generatingRule,
      Artifact artifact,
      ImmutableList<FilesetTraversalParams> traversals) {
    super(label, configurationKey, visibility, artifact, generatingRule);
    FilesetProvider provider = generatingRule.getProvider(FilesetProvider.class);
    Preconditions.checkArgument(provider != null);
    filesetTraversals = traversals;
  }

  @Override
  public ImmutableList<FilesetTraversalParams> getTraversals() {
    return filesetTraversals;
  }
}
